<?php

namespace App\Http\Controllers\Admin;

use App\CentralLogics\Helpers;
use App\Http\Controllers\Controller;
use App\Model\Notification;
use App\Traits\UploadSizeHelper;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class NotificationController extends Controller
{
    use UploadSizeHelper;
    public function __construct(
        private Notification $notification
    ){
        $this->initUploadLimits();
    }

    /**
     * @param Request $request
     * @return Factory|View|Application
     */
    function index(Request $request): View|Factory|Application
    {
        $perPage = (int) $request->query('per_page', Helpers::getPagination());

        $queryParam = ['per_page' => $perPage];
        $search = $request['search'];

        if($request->has('search'))
        {
            $key = explode(' ', $request['search']);
            $notifications = $this->notification->where(function ($q) use ($key) {
                        foreach ($key as $value) {
                            $q->orWhere('title', 'like', "%{$value}%")
                            ->orWhere('description', 'like', "%{$value}%");
                        }
            });
            $queryParam['search'] = $request->search;
;
        }else{
           $notifications = $this->notification;
        }

        $notifications = $notifications->latest()
            ->paginate($perPage)
            ->appends($queryParam);

        return view('admin-views.notification.index', compact('notifications','search', 'perPage'));
    }


    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function store(Request $request): RedirectResponse
    {
        $check = $this->validateUploadedFile($request, ['image']);
        if ($check !== true) {
            return $check;
        }

        $request->validate([
            'title' => 'required|max:255',
            'description' => 'required|max:255',
            'image'=> 'required|mimes:jpg,jpeg,png|max:' . $this->maxImageSizeKB
        ], [
            'title.required' => 'title is required!',
            'image.required' => translate('Image is required!'),
            'images.mimes' => 'Image must be a file of type: jpg, jpeg, png',
            'image.max' => translate('Image size must be below ' . $this->maxImageSizeReadable)
        ]);

        if ($request->has('image')) {
            $imageName = Helpers::upload('notification/', 'png', $request->file('image'));
        } else {
            $imageName = null;
        }

        $notification = $this->notification;
        $notification->title = $request->title;
        $notification->description = $request->description;
        $notification->image = $imageName;
        $notification->status = 1;
        $notification->save();

        try {
            $notification->type = 'general';
            Helpers::send_push_notif_to_topic($notification, 'grofresh');
            Toastr::success(translate('Notification sent successfully!'));
        } catch (\Exception $e) {
            Toastr::warning(translate('Push notification failed!'));
        }

        return back();
    }

    /**
     * @param $id
     * @return Factory|View|Application
     */
    public function edit($id): View|Factory|Application
    {
        $notification = $this->notification->find($id);
        return view('admin-views.notification.edit', compact('notification'));
    }

    /**
     * @param Request $request
     * @param $id
     * @return RedirectResponse
     */
    public function update(Request $request, $id): RedirectResponse
    {
        $check = $this->validateUploadedFile($request, ['image']);
        if ($check !== true) {
            return $check;
        }

        $request->validate([
            'title' => 'required|max:255',
            'description' => 'required|max:255',
            'image'=> 'nullable|mimes:jpg,jpeg,png|max:' . $this->maxImageSizeKB
        ], [
            'title.required' => 'title is required!',
            'image.mimes' => 'Image must be a file of type: jpg, jpeg, png',
            'image.max' => translate('Image size must be below ' . $this->maxImageSizeReadable)
        ]);

        $notification = $this->notification->find($id);

        if ($request->has('image')) {
            $imageName = Helpers::update('notification/', $notification->image, 'png', $request->file('image'));
        } else {
            $imageName = $notification['image'];
        }

        $notification->title = $request->title;
        $notification->description = $request->description;
        $notification->image = $imageName;
        $notification->save();
        Toastr::success(translate('Notification updated successfully!'));
        return back();
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function status(Request $request): \Illuminate\Http\RedirectResponse
    {
        $notification = $this->notification->find($request->id);
        $notification->status = $request->status;
        $notification->save();
        Toastr::success(translate('Notification status updated!'));
        return back();
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function delete(Request $request): \Illuminate\Http\RedirectResponse
    {
        $notification = $this->notification->find($request->id);
        if (Storage::disk('public')->exists('notification/' . $notification['image'])) {
            Storage::disk('public')->delete('notification/' . $notification['image']);
        }
        $notification->delete();
        Toastr::success(translate('Notification removed!'));
        return back();
    }
}
